"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.clearNativeFolder = clearNativeFolder;
exports.hasRequiredAndroidFilesAsync = hasRequiredAndroidFilesAsync;
exports.hasRequiredIOSFilesAsync = hasRequiredIOSFilesAsync;
exports.getMalformedNativeProjectsAsync = getMalformedNativeProjectsAsync;
exports.promptToClearMalformedNativeProjectsAsync = promptToClearMalformedNativeProjectsAsync;
var _configPlugins = require("@expo/config-plugins");
var _chalk = _interopRequireDefault(require("chalk"));
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var Log = _interopRequireWildcard(require("../log"));
var _dir = require("../utils/dir");
var _interactive = require("../utils/interactive");
var _ora = require("../utils/ora");
var _prompts = require("../utils/prompts");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
async function clearNativeFolder(projectRoot, folders) {
    const step = (0, _ora).logNewSection(`Clearing ${folders.join(", ")}`);
    try {
        await Promise.all(folders.map((folderName)=>_fs.default.promises.rm(_path.default.join(projectRoot, folderName), {
                recursive: true,
                force: true
            })
        ));
        step.succeed(`Cleared ${folders.join(", ")} code`);
    } catch (error) {
        step.fail(`Failed to delete ${folders.join(", ")} code: ${error.message}`);
        throw error;
    }
}
async function hasRequiredAndroidFilesAsync(projectRoot) {
    try {
        await Promise.all([
            _configPlugins.AndroidConfig.Paths.getAppBuildGradleAsync(projectRoot),
            _configPlugins.AndroidConfig.Paths.getProjectBuildGradleAsync(projectRoot),
            _configPlugins.AndroidConfig.Paths.getAndroidManifestAsync(projectRoot),
            _configPlugins.AndroidConfig.Paths.getMainApplicationAsync(projectRoot), 
        ]);
        return true;
    } catch  {
        return false;
    }
}
async function hasRequiredIOSFilesAsync(projectRoot) {
    try {
        // If any of the following required files are missing, then the project is malformed.
        await Promise.all([
            _configPlugins.IOSConfig.Paths.getAllXcodeProjectPaths(projectRoot),
            _configPlugins.IOSConfig.Paths.getAllPBXProjectPaths(projectRoot), 
        ]);
        return true;
    } catch  {
        return false;
    }
}
/**
 * Filter out platforms that do not have an existing platform folder.
 * If the user wants to validate that neither of ['ios', 'android'] are malformed then we should
 * first check that both `ios` and `android` folders exist.
 *
 * This optimization prevents us from prompting to clear a "malformed" project that doesn't exist yet.
 */ async function filterPlatformsThatDoNotExistAsync(projectRoot, platforms) {
    const valid = await Promise.all(platforms.map(async (platform)=>{
        if (await (0, _dir).directoryExistsAsync(_path.default.join(projectRoot, platform))) {
            return platform;
        }
        return null;
    }));
    return valid.filter(Boolean);
}
async function getMalformedNativeProjectsAsync(projectRoot, platforms) {
    const VERIFIERS = {
        android: hasRequiredAndroidFilesAsync,
        ios: hasRequiredIOSFilesAsync
    };
    const checkPlatforms = await filterPlatformsThatDoNotExistAsync(projectRoot, platforms);
    return (await Promise.all(checkPlatforms.map(async (platform)=>{
        if (await VERIFIERS[platform](projectRoot)) {
            return false;
        }
        return platform;
    }))).filter(Boolean);
}
async function promptToClearMalformedNativeProjectsAsync(projectRoot, checkPlatforms) {
    const platforms = await getMalformedNativeProjectsAsync(projectRoot, checkPlatforms);
    if (!platforms.length) {
        return;
    }
    const displayPlatforms = platforms.map((platform)=>_chalk.default.cyan(platform)
    );
    // Prompt which platforms to reset.
    const message = platforms.length > 1 ? `The ${displayPlatforms[0]} and ${displayPlatforms[1]} projects are malformed` : `The ${displayPlatforms[0]} project is malformed`;
    if (// If the process is non-interactive, default to clearing the malformed native project.
    // This would only happen on re-running eject.
    !(0, _interactive).isInteractive() || // Prompt to clear the native folders.
    (await (0, _prompts).confirmAsync({
        message: `${message}, would you like to clear the project files and reinitialize them?`,
        initial: true
    }))) {
        if (!(0, _interactive).isInteractive()) {
            Log.warn(`${message}, project files will be cleared and reinitialized.`);
        }
        await clearNativeFolder(projectRoot, platforms);
    } else {
        // Warn the user that the process may fail.
        Log.warn("Continuing with malformed native projects");
    }
}

//# sourceMappingURL=clearNativeFolder.js.map