"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var PackageManager = _interopRequireWildcard(require("@expo/package-manager"));
var _requireg = _interopRequireDefault(require("requireg"));
var _resolveFrom = _interopRequireDefault(require("resolve-from"));
var _semver = _interopRequireDefault(require("semver"));
var Log = _interopRequireWildcard(require("../../../log"));
var _delay = require("../../../utils/delay");
var _env = require("../../../utils/env");
var _errors = require("../../../utils/errors");
var _prompts = require("../../../utils/prompts");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
var _ref;
const debug = require("debug")("expo:doctor:externalModule");
class ExternalModuleVersionError extends _errors.CommandError {
    constructor(message, shouldGloballyInstall){
        super("EXTERNAL_MODULE_VERSION", message);
        this.shouldGloballyInstall = shouldGloballyInstall;
    }
}
exports.ExternalModuleVersionError = ExternalModuleVersionError;
class ExternalModule {
    constructor(projectRoot, pkg, promptMessage){
        this.projectRoot = projectRoot;
        this.pkg = pkg;
        this.promptMessage = promptMessage;
        this.instance = null;
    }
    /** Resolve the globally or locally installed instance, or prompt to install. */ async resolveAsync({ prefersGlobalInstall , ...options } = {}) {
        try {
            var ref;
            return (ref = this.getVersioned()) != null ? ref : this.installAsync({
                ...options,
                shouldGloballyInstall: prefersGlobalInstall
            });
        } catch (error) {
            if (error instanceof ExternalModuleVersionError) {
                var _shouldGloballyInstall;
                // If the module version in not compliant with the version range,
                // we should prompt the user to install the package where it already exists.
                return this.installAsync({
                    ...options,
                    shouldGloballyInstall: (_shouldGloballyInstall = error.shouldGloballyInstall) != null ? _shouldGloballyInstall : prefersGlobalInstall
                });
            }
            throw error;
        }
    }
    /** Prompt the user to install the package and try again. */ async installAsync({ shouldPrompt =true , autoInstall , shouldGloballyInstall  } = {}) {
        const packageName = [
            this.pkg.name,
            this.pkg.versionRange
        ].join("@");
        if (!autoInstall) {
            // Delay the prompt so it doesn't conflict with other dev tool logs
            await (0, _delay).delayAsync(100);
        }
        const answer = autoInstall || shouldPrompt && await (0, _prompts).confirmAsync({
            message: this.promptMessage(packageName),
            initial: true
        });
        if (answer) {
            Log.log(`Installing ${packageName}...`);
            // Always use npm for global installs
            const packageManager = shouldGloballyInstall ? new PackageManager.NpmPackageManager({
                cwd: this.projectRoot,
                log: Log.log,
                silent: !_env.env.EXPO_DEBUG
            }) : PackageManager.createForProject(this.projectRoot, {
                silent: !_env.env.EXPO_DEBUG
            });
            try {
                if (shouldGloballyInstall) {
                    await packageManager.addGlobalAsync([
                        packageName
                    ]);
                } else {
                    await packageManager.addDevAsync([
                        packageName
                    ]);
                }
                Log.log(`Installed ${packageName}`);
            } catch (error) {
                error.message = `Failed to install ${packageName} ${shouldGloballyInstall ? "globally" : "locally"}: ${error.message}`;
                throw error;
            }
            return await this.resolveAsync({
                shouldPrompt: false
            });
        }
        throw new _errors.CommandError("EXTERNAL_MODULE_AVAILABILITY", `Please install ${packageName} and try again`);
    }
    /** Get the module. */ get() {
        try {
            return this.getVersioned();
        } catch  {
            return null;
        }
    }
    /** Get the module, throws if the module is not versioned correctly. */ getVersioned() {
        var ref, _instance;
        (_instance = (_ref = this).instance) != null ? _instance : _ref.instance = (ref = this._resolveModule(true)) != null ? ref : this._resolveModule(false);
        return this.instance;
    }
    /** Exposed for testing. */ _require(moduleId) {
        return require(moduleId);
    }
    /** Resolve a copy that's installed in the project. Exposed for testing. */ _resolveLocal(moduleId) {
        return (0, _resolveFrom).default(this.projectRoot, moduleId);
    }
    /** Resolve a copy that's installed globally. Exposed for testing. */ _resolveGlobal(moduleId) {
        return _requireg.default.resolve(moduleId);
    }
    /** Resolve the module and verify the version. Exposed for testing. */ _resolveModule(isLocal) {
        const resolver = isLocal ? this._resolveLocal.bind(this) : this._resolveGlobal.bind(this);
        try {
            const packageJsonPath = resolver(`${this.pkg.name}/package.json`);
            const packageJson = this._require(packageJsonPath);
            if (packageJson) {
                if (_semver.default.satisfies(packageJson.version, this.pkg.versionRange)) {
                    const modulePath = resolver(this.pkg.name);
                    const requiredModule = this._require(modulePath);
                    if (requiredModule == null) {
                        throw new _errors.CommandError("EXTERNAL_MODULE_EXPORT", `${this.pkg.name} exports a nullish value, which is not allowed.`);
                    }
                    return requiredModule;
                }
                throw new ExternalModuleVersionError(`Required module '${this.pkg.name}@${packageJson.version}' does not satisfy ${this.pkg.versionRange}. Installed at: ${packageJsonPath}`, !isLocal);
            }
        } catch (error) {
            if (error instanceof _errors.CommandError) {
                throw error;
            } else if (error.code !== "MODULE_NOT_FOUND") {
                debug("Failed to resolve module", error.message);
            }
        }
        return null;
    }
}
exports.ExternalModule = ExternalModule;

//# sourceMappingURL=ExternalModule.js.map