"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _promises = _interopRequireDefault(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var Log = _interopRequireWildcard(require("../../../log"));
var _dir = require("../../../utils/dir");
var _env = require("../../../utils/env");
var _glob = require("../../../utils/glob");
var _prerequisite = require("../Prerequisite");
var _ensureDependenciesAsync = require("../dependencies/ensureDependenciesAsync");
var _updateTSConfig = require("./updateTSConfig");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
const debug = require("debug")("expo:doctor:typescriptSupport");
class TypeScriptProjectPrerequisite extends _prerequisite.ProjectPrerequisite {
    /** Ensure a project that hasn't explicitly disabled web support has all the required packages for running in the browser. */ async assertImplementation() {
        if (_env.env.EXPO_NO_TYPESCRIPT_SETUP) {
            Log.warn("Skipping TypeScript setup: EXPO_NO_TYPESCRIPT_SETUP is enabled.");
            return;
        }
        debug("Ensuring TypeScript support is setup");
        const tsConfigPath = _path.default.join(this.projectRoot, "tsconfig.json");
        // Ensure the project is TypeScript before continuing.
        const intent = await this._getSetupRequirements();
        if (!intent) {
            return;
        }
        // Ensure TypeScript packages are installed
        await this._ensureDependenciesInstalledAsync();
        // Update the config
        await (0, _updateTSConfig).updateTSConfigAsync({
            tsConfigPath,
            isBootstrapping: intent.isBootstrapping
        });
    }
    /** Exposed for testing. */ async _getSetupRequirements() {
        const tsConfigPath = await this._hasTSConfig();
        // Enable TS setup if the project has a `tsconfig.json`
        if (tsConfigPath) {
            const content = await _promises.default.readFile(tsConfigPath, {
                encoding: "utf8"
            }).then((txt)=>txt.trim()
            , // null when the file doesn't exist.
            ()=>null
            );
            const isBlankConfig = content === "" || content === "{}";
            return {
                isBootstrapping: isBlankConfig
            };
        }
        // This is a somewhat heavy check in larger projects.
        // Test that this is reasonably paced by running expo start in `expo/apps/native-component-list`
        const typescriptFile = await this._queryFirstTypeScriptFileAsync();
        if (typescriptFile) {
            return {
                isBootstrapping: true
            };
        }
        return null;
    }
    /** Exposed for testing. */ async _ensureDependenciesInstalledAsync({ exp  } = {}) {
        try {
            return await (0, _ensureDependenciesAsync).ensureDependenciesAsync(this.projectRoot, {
                exp,
                installMessage: `It looks like you're trying to use TypeScript but don't have the required dependencies installed.`,
                warningMessage: "If you're not using TypeScript, please remove the TypeScript files from your project",
                requiredPackages: [
                    // use typescript/package.json to skip node module cache issues when the user installs
                    // the package and attempts to resolve the module in the same process.
                    {
                        file: "typescript/package.json",
                        pkg: "typescript"
                    },
                    {
                        file: "@types/react/package.json",
                        pkg: "@types/react"
                    }, 
                ]
            });
        } catch (error) {
            // Reset the cached check so we can re-run the check if the user re-runs the command by pressing 'w' in the Terminal UI.
            this.resetAssertion();
            throw error;
        }
    }
    /** Return the first TypeScript file in the project. */ async _queryFirstTypeScriptFileAsync() {
        const results = await (0, _glob).wrapGlobWithTimeout(()=>// TODO(Bacon): Use `everyMatch` since a bug causes `anyMatch` to return inaccurate results when used multiple times.
            (0, _glob).everyMatchAsync("**/*.@(ts|tsx)", {
                cwd: this.projectRoot,
                ignore: [
                    "**/@(Carthage|Pods|node_modules)/**",
                    "**/*.d.ts",
                    "@(ios|android|web|web-build|dist)/**", 
                ]
            })
        , 5000);
        if (results === false) {
            return null;
        }
        var ref;
        return (ref = results[0]) != null ? ref : null;
    }
    async _hasTSConfig() {
        const tsConfigPath = _path.default.join(this.projectRoot, "tsconfig.json");
        if (await (0, _dir).fileExistsAsync(tsConfigPath)) {
            return tsConfigPath;
        }
        return null;
    }
}
exports.TypeScriptProjectPrerequisite = TypeScriptProjectPrerequisite;

//# sourceMappingURL=TypeScriptProjectPrerequisite.js.map