"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _configPlugins = require("@expo/config-plugins");
var _accepts = _interopRequireDefault(require("accepts"));
var _assert = _interopRequireDefault(require("assert"));
var _crypto = _interopRequireDefault(require("crypto"));
var _formData = _interopRequireDefault(require("form-data"));
var _structuredHeaders = require("structured-headers");
var _getProject = require("../../../api/getProject");
var _settings = require("../../../api/settings");
var _signManifest = require("../../../api/signManifest");
var _userSettings = _interopRequireDefault(require("../../../api/user/UserSettings"));
var _user = require("../../../api/user/user");
var _rudderstackClient = require("../../../utils/analytics/rudderstackClient");
var _codesigning = require("../../../utils/codesigning");
var _errors = require("../../../utils/errors");
var _fn = require("../../../utils/fn");
var _url = require("../../../utils/url");
var _manifestMiddleware = require("./ManifestMiddleware");
var _resolvePlatform = require("./resolvePlatform");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const debug = require("debug")("expo:start:server:middleware:ExpoGoManifestHandlerMiddleware");
class ExpoGoManifestHandlerMiddleware extends _manifestMiddleware.ManifestMiddleware {
    getParsedHeaders(req) {
        let platform = (0, _resolvePlatform).parsePlatformHeader(req);
        if (!platform) {
            debug(`No "expo-platform" header or "platform" query parameter specified. Falling back to "none".`);
            platform = "none";
        }
        (0, _resolvePlatform).assertRuntimePlatform(platform);
        // Expo Updates clients explicitly accept "multipart/mixed" responses while browsers implicitly
        // accept them with "accept: */*". To make it easier to debug manifest responses by visiting their
        // URLs in a browser, we denote the response as "text/plain" if the user agent appears not to be
        // an Expo Updates client.
        const accept = (0, _accepts).default(req);
        const explicitlyPrefersMultipartMixed = accept.types([
            "unknown/unknown",
            "multipart/mixed"
        ]) === "multipart/mixed";
        const expectSignature = req.headers["expo-expect-signature"];
        return {
            explicitlyPrefersMultipartMixed,
            platform,
            acceptSignature: !!req.headers["expo-accept-signature"],
            expectSignature: expectSignature ? String(expectSignature) : null,
            hostname: (0, _url).stripPort(req.headers["host"])
        };
    }
    getDefaultResponseHeaders() {
        const headers = new Map();
        // set required headers for Expo Updates manifest specification
        headers.set("expo-protocol-version", 0);
        headers.set("expo-sfv-version", 0);
        headers.set("cache-control", "private, max-age=0");
        return headers;
    }
    async _getManifestResponseAsync(requestOptions) {
        var ref, ref1;
        const { exp , hostUri , expoGoConfig , bundleUrl  } = await this._resolveProjectSettingsAsync(requestOptions);
        var _runtimeVersion;
        const runtimeVersion = _configPlugins.Updates.getRuntimeVersion({
            ...exp,
            runtimeVersion: (_runtimeVersion = exp.runtimeVersion) != null ? _runtimeVersion : {
                policy: "sdkVersion"
            }
        }, requestOptions.platform);
        if (!runtimeVersion) {
            throw new _errors.CommandError("MANIFEST_MIDDLEWARE", `Unable to determine runtime version for platform '${requestOptions.platform}'`);
        }
        const codeSigningInfo = await (0, _codesigning).getCodeSigningInfoAsync(exp, requestOptions.expectSignature, this.options.privateKeyPath);
        const easProjectId = (ref = exp.extra) == null ? void 0 : (ref1 = ref.eas) == null ? void 0 : ref1.projectId;
        const shouldUseAnonymousManifest = await shouldUseAnonymousManifestAsync(easProjectId, codeSigningInfo);
        const userAnonymousIdentifier = await _userSettings.default.getAnonymousIdentifierAsync();
        if (!shouldUseAnonymousManifest) {
            (0, _assert).default(easProjectId);
        }
        const scopeKey = shouldUseAnonymousManifest ? `@${_user.ANONYMOUS_USERNAME}/${exp.slug}-${userAnonymousIdentifier}` : await this.getScopeKeyForProjectIdAsync(easProjectId);
        const expoUpdatesManifest = {
            id: _crypto.default.randomUUID(),
            createdAt: new Date().toISOString(),
            runtimeVersion,
            launchAsset: {
                key: "bundle",
                contentType: "application/javascript",
                url: bundleUrl
            },
            assets: [],
            metadata: {},
            extra: {
                eas: {
                    projectId: easProjectId != null ? easProjectId : undefined
                },
                expoClient: {
                    ...exp,
                    hostUri
                },
                expoGo: expoGoConfig,
                scopeKey
            }
        };
        const headers = this.getDefaultResponseHeaders();
        if (requestOptions.acceptSignature && !shouldUseAnonymousManifest) {
            const manifestSignature = await this.getSignedManifestStringAsync(expoUpdatesManifest);
            headers.set("expo-manifest-signature", manifestSignature);
        }
        const stringifiedManifest = JSON.stringify(expoUpdatesManifest);
        let manifestPartHeaders = null;
        let certificateChainBody = null;
        if (codeSigningInfo) {
            const signature = (0, _codesigning).signManifestString(stringifiedManifest, codeSigningInfo);
            manifestPartHeaders = {
                "expo-signature": (0, _structuredHeaders).serializeDictionary(convertToDictionaryItemsRepresentation({
                    keyid: codeSigningInfo.keyId,
                    sig: signature,
                    alg: "rsa-v1_5-sha256"
                }))
            };
            certificateChainBody = codeSigningInfo.certificateChainForResponse.join("\n");
        }
        const form = this.getFormData({
            stringifiedManifest,
            manifestPartHeaders,
            certificateChainBody
        });
        headers.set("content-type", requestOptions.explicitlyPrefersMultipartMixed ? `multipart/mixed; boundary=${form.getBoundary()}` : "text/plain");
        return {
            body: form.getBuffer().toString(),
            version: runtimeVersion,
            headers
        };
    }
    getFormData({ stringifiedManifest , manifestPartHeaders , certificateChainBody  }) {
        const form = new _formData.default();
        form.append("manifest", stringifiedManifest, {
            contentType: "application/json",
            header: {
                ...manifestPartHeaders
            }
        });
        if (certificateChainBody && certificateChainBody.length > 0) {
            form.append("certificate_chain", certificateChainBody, {
                contentType: "application/x-pem-file"
            });
        }
        return form;
    }
    trackManifest(version) {
        (0, _rudderstackClient).logEventAsync("Serve Expo Updates Manifest", {
            runtimeVersion: version
        });
    }
    getSignedManifestStringAsync = (0, _fn).memoize(_signManifest.signExpoGoManifestAsync);
    getScopeKeyForProjectIdAsync = (0, _fn).memoize(getScopeKeyForProjectIdAsync);
}
exports.ExpoGoManifestHandlerMiddleware = ExpoGoManifestHandlerMiddleware;
/**
 * 1. No EAS project ID in config, then use anonymous scope key
 * 2. When offline or not logged in
 *   a. If code signing not accepted by client (only legacy manifest signing is supported), then use anonymous scope key
 *   b. If code signing accepted by client and no development code signing certificate is cached, then use anonymous scope key
 */ async function shouldUseAnonymousManifestAsync(easProjectId, codeSigningInfo) {
    if (!easProjectId || _settings.APISettings.isOffline && codeSigningInfo === null) {
        return true;
    }
    return !await (0, _user).getUserAsync();
}
async function getScopeKeyForProjectIdAsync(projectId) {
    const project = await (0, _getProject).getProjectAsync(projectId);
    return project.scopeKey;
}
function convertToDictionaryItemsRepresentation(obj) {
    return new Map(Object.entries(obj).map(([k, v])=>{
        return [
            k,
            [
                v,
                new Map()
            ]
        ];
    }));
}

//# sourceMappingURL=ExpoGoManifestHandlerMiddleware.js.map