let ac = new AudioContext()

const master = new DynamicsCompressorNode(ac, {
    threshold: -50,  // in dB
    knee: 40,        // in dB
    ratio: 12,       // ratio
    attack: 0,       // in seconds
    release: 0.25    // in seconds
});

master.connect(ac.destination)

let octaveCharacters = {"c": 0, "d": 2, "e": 4, "f": 5, "g": 7, "a": 9, "b": 11}

function parseNote(noteString) {
    let [note, octave] = noteString.split("-")
    let [key, accidental] = Array.from(note)
    let keyNumber = octaveCharacters[key.toLowerCase()]
    let shift = 0
    if (accidental && accidental == "#" || accidental == "b") {
        shift = accidental == "#" ? 1 : -1
    }
    let noteNumber = keyNumber + shift + octave * 12
    return noteNumber
}

const resumeIfSuspended = ac => {
    if (ac.state == "suspended")
        ac.resume()
}

keyActions = {
    p() {
        resumeIfSuspended(ac)
        playSong()
    },
}

document.getElementById("start-button").addEventListener("click", function(e){
    resumeIfSuspended(ac)
    playSong()
    this.style.display = "none"
})

document.body.addEventListener("keydown", function (e) {
    if (keyActions[e.key]) {
        keyActions[e.key](e)
    }
    if (patterns[e.key]) {
        playPattern(e.key, 0, true)
    }
    if (sounds[e.key]) {
        sounds[e.key](0, false, 0)
    }
})

const BPM = 160
const stepSeconds = 60 / BPM / 4

const patternOscillators = {}

let textCount = 0

const instruments = {
    thump({
            frequency = 440, 
            detune = 0,
            detuneValues = [0, 0],
            noiseValues = [0.5, 0],
            time = 0,  
            vol = 0.1, 
            noiseVol = 0.0, 
            noiseDecay = 0.125, 
            decay = 0.125,
            visCallback = defaultAnim, 
            patternKey = "0", 
            key,
            octaveDouble = false
        } = params) {

        let osci = new OscillatorNode(ac, {
            type: "sine",
            frequency
        })

        const visOffset = 1 / 60
        time += visOffset

        let gaini = new GainNode(ac, { gain: vol })

        const startTime = Math.max(ac.currentTime, ac.currentTime + time)
        const endTime = startTime + decay
        const endTimeNoise = startTime + noiseDecay
        osci.start(startTime)

        if(!patternOscillators[patternKey]){
            patternOscillators[patternKey] = []   
        }

        patternOscillators[patternKey].push({
            osc: osci,
            gain: gaini,
            startTime
        })
        
        // osci.detune.setValueAtTime(0, startTime)
        // osci.detune.linearRampToValueAtTime(detuneTarget, endTime)

        osci.detune.setValueAtTime(0, startTime)
        osci.detune.setValueCurveAtTime(detuneValues, startTime, endTime-startTime)

        gaini.gain.setValueAtTime(vol, startTime)
        gaini.gain.linearRampToValueAtTime(0, endTime)

        osci.stop(endTime)

        osci.onended = function(){
            patternOscillators[patternKey] = patternOscillators[patternKey].filter(oscItem => oscItem.osc !== osci) 
        }

        let vis = new ConstantSourceNode(ac, { offset: 0 })
        vis.start(startTime - visOffset)
        vis.stop(startTime)
        vis.onended = () => visCallback(key)

        osci
            .connect(gaini)
            .connect(master)

        if(noiseVol > 0){
            let noiseGain = new GainNode(ac, { gain: vol })

            var bufferSize = ac.sampleRate / 4,
            noiseBuffer = ac.createBuffer(1, bufferSize, ac.sampleRate),
            output = noiseBuffer.getChannelData(0);
            for (var i = 0; i < bufferSize; i++) {
                output[i] = (Math.random() * 2 - 1) * 0.25;
            }
            
            var whiteNoise = ac.createBufferSource();
            whiteNoise.buffer = noiseBuffer;
            whiteNoise.loop = true;
            whiteNoise.start(startTime);
            whiteNoise.stop(endTime)

            noiseGain.gain.setValueAtTime(vol, startTime)
            noiseGain.gain.setValueCurveAtTime(noiseValues, startTime, noiseDecay)
            
            whiteNoise
                .connect(noiseGain)
                .connect(master)

            patternOscillators[patternKey].push({
                osc: whiteNoise,
                gain: noiseGain,
                startTime
            })

            whiteNoise.onended = function(){
                patternOscillators[patternKey] = patternOscillators[patternKey].filter(oscItem => oscItem.osc !== whiteNoise) 
            }
        }
    },
    vis({
        time = 0,
        decay = 0.125,
        visCallback = defaultAnim, 
        patternKey = "0", 
        key
    } = params) {
        const visOffset = 1 / 60
        time += visOffset

        const startTime = Math.max(ac.currentTime, ac.currentTime + time)
        const endTime = startTime + decay
        
        let vis = new ConstantSourceNode(ac, { offset: 0 })
        vis.start(startTime - visOffset)
        vis.stop(startTime)
        vis.onended = () => visCallback(key)
    }
}

const transformAnim = [
    { 
        fontSize: "40vh",
        lineHeight: "40vh",
        width: "40vh",
        height: "40vh",
        filter: "grayscale(0%)"
    },
    {
        fontSize: "20vh",
        lineHeight: "20vh",
        width: "20vh",
        height: "20vh",
        filter: "grayscale(100%)"
    }
]

const animDuration = {
    duration: 100,
    iterations: 1,
}

function defaultAnim(key){
    [...document.getElementsByClassName(key)].forEach(elem => {
        elem.animate(
            transformAnim,
            animDuration
        );
    })
}

/*
visCallback: () => {
    document.getElementById(key).animate(
        transformAnim,
        animDuration
    );
},
*/

const textElements = [...document.getElementsByClassName("text-middle")]

function createDetuneEnvelope(str, min, max){
    let values = [...str].map(char => {
        let normalized = ((char.charCodeAt(0) - 9601) / 7)
        return normalized * (max-min) + min
    })
    return values
}

function createVolEnvelope(str){
    let values = [...str].map(char => {
        let normalized = ((char.charCodeAt(0) - 9601) / 7)
        return normalized
    })
    return values
}

const sounds = {
    q(time, accent, patternKey) {
        const key = arguments.callee.name
        instruments.thump({
            frequency: 80,
            time,
            detuneValues: createDetuneEnvelope(
                "█▃▂▂▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁", 
                -1200, 
                4800
            ),
            vol: 0.2 + accent * 0.05,
            noiseVol: 0.0,
            decay: 0.5,
            visCallback: () => {
                textElements.forEach(elem => {
                    elem.animate([
                        { 
                            fontVariationSettings: "'wght' 1200, 'wdth' 100",
                            color: "#FFF"
                        },
                        { 
                            fontVariationSettings: "'wght' 100, 'wdth' 10",
                            color: "#333"
                        }
                    ], { duration: 100, iterations: 1 });
                });
                [...document.getElementsByClassName(key)].forEach(elem => {
                    elem.animate(
                        transformAnim,
                        animDuration
                    );
                })
            },
            patternKey,
            key
        })
    },
    w(time, accent, patternKey) {
        const key = arguments.callee.name
        instruments.thump({
            frequency: 160,
            detuneValues: createDetuneEnvelope(
                "▁▁▁▁▁▁▁", 
                0, 
                -200
            ),
            noiseValues: createVolEnvelope("█▄▂▂▁▁▁▁▁"),
            time,
            vol: 0.2 + accent * 0.05,
            noiseVol: 0.5 + accent * 0.05,
            decay: 0.08,
            noiseDecay: 0.2,
            visCallback: defaultAnim,
            patternKey,
            key
        })
    },
    e(time, accent, patternKey) {
        const key = arguments.callee.name
        instruments.thump({
            frequency: 880,
            time,
            vol: 0.02 + accent * 0.03,
            decay: 0.01,
            visCallback: defaultAnim,
            patternKey,
            key
        })
    },
    r(time, accent, patternKey) {
        const key = arguments.callee.name
        instruments.thump({
            frequency: 1760,
            time,
            vol: 0,
            decay: 0.02,
            noiseVol: 0.001 + accent * 0.001,
            noiseValues: createVolEnvelope("▁▂█▄▂"),
            visCallback: defaultAnim,
            patternKey,
            key
        })
    },
    $(time, accent, patternKey, index) {
        const key = arguments.callee.name
        instruments.vis({
            time,
            decay: 0.01,
            visCallback: () => {
                let textLines = texts.trim().split("\n")
                let message = textLines[index % textLines.length]
                speak({
                    voice: fredVoice,
                    message,
                    volume: 0.2,
                    rate: 1.2,
                    pitch: 1.0
                })
                textElements.forEach(elem => elem.innerHTML = message)
            },
            patternKey,
            key
        })
    },
    "&"(time, accent, patternKey, index) {
        const key = arguments.callee.name
        instruments.vis({
            time,
            decay: 0.01,
            visCallback: () => {
                document.body.style.backgroundColor = "#000"
            },
            patternKey,
            key
        })
    },
    "/"(time, accent, patternKey, index) {
        const key = arguments.callee.name
        instruments.vis({
            time,
            decay: 0.01,
            visCallback: () => {
                document.body.style.backgroundColor = "#FF0";
                [...document.getElementsByClassName("emoji-small")].forEach(
                    emoji => emoji.classList.remove("hidden")
                )
            },
            patternKey,
            key
        })
    },
    "å"(time, accent, patternKey, index) {
        const key = arguments.callee.name
        instruments.vis({
            time,
            decay: 0.01,
            visCallback: () => {
                document.getElementsByClassName("scene")[0].classList.add("fade")
            },
            patternKey,
            key
        })
    },
}

let patterns = {
a: `
..ee
Qq.r
`,
d: `
Q.rr .rQ. Wrr. Qrrr .rq. Qrr. Wrr. r.r.
`,
s: `
We..
..qq
`,
f: `
WWww EEWW wwEE WWww
`,
g: `
rrRR rrRR rrRR rrRR
`,

"!": `
$... ..$. .... $... .... $... $... ....
.... .... .... .... .... .... .... ....
.... .... .... .... .... .... .... ....
`,
"#": `
$... ..$. .... $... .... $... $... ....
W... ..w. w... w... ..w. w... w... ....
.... .... .... .... .... .... .... ....
`,
"€": `
$... ..$. .... $... .... $... $... ....
w..w ..w. .w.. w.w. WWww wwWW WWww WWWW
.... .... .... .... .... .... .... ....
`,
x: `
$... ..$. .... $... .... $... $... ....
..e. e... ..e. ..e. e... ..e. ..e. e.e.
q... ..q. w... q... ..q. q... w... ....
`,
c: `
$... ..$. .... $... .... $... $... ....
..ee .e.. .ee. ..ee .ee. ..e. .ee. .ee.
q... ..q. w... q... ..q. q... w... ....
/... .... .... .... .... .... .... ....
`,
v: `
$... ..$. .... $... .... $... $... ....
..ee .e.. .ee. ..ee .ee. ..e. .ee. .ee.
q..q ..q. w.q. q..q .qq. q... w... ....
rrrr rr.r r.rr .rr. rrrr rr.r r.rr rrrr 
&... .... .... .... .... .... .... ....
`,


_: `
....
`,

"=": `
å...
`,
}

let texts = `
I
still
love
too
ling
I
still
love
too
ling
I
still
love
too
ling
I
still
love
too
ling
and
c
s
s
and
no
dry
stuff
and
i
still
love
lamp
`

let song = "_ !!!# xxx# ccc€ vvvv !!!# xxx# ccc€ vvv€ vvv€="

function stopAll(patternKey, currentTime) {
    if (patternOscillators[patternKey] && patternOscillators[patternKey].length) {
        patternOscillators[patternKey].forEach(oscItem => {
            if (oscItem.startTime <= currentTime) {
                oscItem.gain.gain.linearRampToValueAtTime(0, currentTime)
                oscItem.osc.stop(currentTime);
            }
        });
        patternOscillators[patternKey] = patternOscillators[patternKey].filter(oscItem => oscItem.startTime > currentTime);
    }
}

function playPattern(patternKey, timeOffset, cancelPrevious = false){
    const pattern = patterns[patternKey]
    seq = pattern.trim().replaceAll(" ", "")
    const patternLength = seq.split("\n")[0].length
    if(cancelPrevious){
        stopAll(patternKey, ac.currentTime + timeOffset)
    }
    seq.split("\n").forEach((line, row) => {
        [...line].forEach((cmd, col) => {
            let time = col * stepSeconds + timeOffset
            if (sounds[cmd.toLowerCase()]) {
                let accent = cmd != cmd.toLowerCase()
                
                if(cmd.toLowerCase() == "$"){
                    sounds[cmd.toLowerCase()](time - stepSeconds, accent, patternKey, textCount)
                    textCount+=1
                } else {
                    sounds[cmd.toLowerCase()](time, accent, patternKey)
                }
            }
            // nested pattern
            else if(patterns[cmd.toLowerCase()]){
                playPattern(cmd.toLowerCase(), time, true)
            }
        })
    })
    return patternLength * stepSeconds
}

function playSong() {
    let patternTimeOffset = 0;
    [...song.replaceAll(" ", "")].forEach((char, patternIndex) => {
        if(patterns[char]){
            patternTimeOffset += playPattern(char, patternTimeOffset)
        }
    })
}

const randomRange = (min, max) => {
    return Math.random() * (max - min) + min
}

let fredVoice;

const container = document.getElementsByClassName("container")[0]

window.onload = function(){
    if (window.speechSynthesis) {
        window.speechSynthesis.onvoiceschanged = () => {
            const voices = window.speechSynthesis.getVoices();
            fredVoice = voices.filter(voice => voice.name == "Fred")[0]
        }
    }
    const emojis = [...document.getElementsByClassName("emoji")]

    emojis.forEach(emoji => {
        emoji.style = `
            --xPos: ${randomRange(-4, 4)}; 
            --yPos: ${randomRange(-4, 4)};  
            --zPos: ${randomRange(-4, 4)};`
    })

}

function pickEmoji(){
    emojis = [..."🪘🥁🪇🪈🛎️🦜🖲️💎🔔⚪️🔴🟠🔺🔷🟦"]
    return emojis[Math.floor(Math.random()*emojis.length)]
}

function generateSphereElements(emojiCount, sphereRadius) {
    const elements = [];

    for (let i = 0; i < emojiCount; i++) {
        const phi = Math.acos(2 * i / emojiCount - 1) - Math.PI / 2; // range from -PI/2 to PI/2
        const theta = Math.sqrt(emojiCount * Math.PI) * phi;

        const x = sphereRadius * Math.cos(phi) * Math.cos(theta);
        const y = sphereRadius * Math.cos(phi) * Math.sin(theta);
        const z = sphereRadius * Math.sin(phi);

        const emoji = document.createElement('span');
        emoji.innerHTML = pickEmoji()
        emoji.classList.add('emoji-small');
        emoji.classList.add('hidden');
        emoji.style = `
            --xPos: ${x}; 
            --yPos: ${y};  
            --zPos: ${z};
        `;

        // Add to the elements array
        elements.push(emoji);
        container.appendChild(emoji)
    }
    return elements;
}

generateSphereElements(50, 10)

// speak(fredVoice, "haloo")



function speak({voice = null, message = "",volume = 1.0,rate= 1.0,pitch=1.0} = {}) {
    let speech = new SpeechSynthesisUtterance(message);
    speech.voice = voice;
    speech.volume = volume;
    speech.rate = rate;
    speech.pitch = pitch;
    window.speechSynthesis.speak(speech);
}

/*
🪘🥁🪇🪈🛎️🦜🖲️💎🔔⚪️🔴🟠🔺🔷🟦
*/

let vals = "▁▂▃▄▅▆▇█";
// 9601-9608
// ▉▊▋▌▍▎▏


/*

.animate([
  { offset: 0, '--ass-border-opacity': 1 },
  { offset: 0.5, '--ass-border-opacity': 0 },
  { offset: 1, '--ass-border-opacity': 1 },
], { duration: 1000, iterations: Infinity });

*/