#ifndef PIPELINE_VERTEX_GLSL
#define PIPELINE_VERTEX_GLSL
// pipeline/vertex.glsl — shared vertex core (keep effects tiny)

// ---- Glue & constants ----
#pragma include "../p3d/in.glsl"
#pragma include "../p3d/uniform.glsl"
#pragma include "../lib/constants.glsl"
#pragma include "../lib/rot.glsl"      // for rot(vec3, vec3) used internally

// ---- Shared uniforms ----
uniform int   u_NumberOfLights;
uniform int   u_InstanceCount;
uniform float u_Scale;
uniform vec3  u_VertPositionOffset;
uniform float u_Time;
uniform float u_Progress;
uniform bool  u_Sandbox;

// ---- Varyings ----
out vec4 vertColor;
out vec4 vertPosition;  // view-space position

#ifndef NO_NORMAL
out vec3 vertNormal;  // view-space normal
#endif

#ifndef NO_TANGENT
out vec3 tangent;  // view-space tangent
#endif

#ifndef NO_UV
out vec2 vertMultiTexCoord0;
#endif

#ifndef NO_SHADOWS
out vec4 vertInShadowSpaces[MAX_LIGHTS];
#endif

#ifdef OPOS
out vec3 opos;
#endif

// ---- Small helpers (one-word) ----

// safe helpers as functions
float t_fn() { return u_Sandbox ? osg_FrameTime : u_Time; }
float p_fn() { return u_Sandbox ? fract(osg_FrameTime * 0.1) : u_Progress; }
float p2_fn() { return u_Sandbox ? fract(osg_FrameTime * 0.02) : u_Progress; }
float p5_fn() { return u_Sandbox ? fract(osg_FrameTime * 0.05) : u_Progress; }
int   sid_fn() { return gl_InstanceID; }  // seed = instance ID

// aliases for convenience (use like variables: t, p, s)
#define t t_fn()
#define p p_fn()
#define p2 p2_fn()
#define p5 p5_fn()
#define sid sid_fn()

// ---- Per-vertex state ----
struct Vertex {
    vec4 position;  // object-space
    vec4 color;
    vec3 offset;  // object-space
    vec3 angle;  // radians XYZ, object-space
};

// Initialize from engine inputs
void begin(inout Vertex v) {
    v.position = (p3d_Vertex + vec4(u_VertPositionOffset, 0.0)) * u_Scale;
    v.color    = p3d_Color;
    v.offset   = vec3(0.0);
    v.angle    = vec3(0.0);
}

// Object-space rotation matrix (XYZ order)
mat3 rotm(in vec3 a) {
    float cx = cos(a.x), sx = sin(a.x);
    float cy = cos(a.y), sy = sin(a.y);
    float cz = cos(a.z), sz = sin(a.z);
    mat3 Rx = mat3( 1,0,0,  0,cx,-sx,  0,sx,cx );
    mat3 Ry = mat3( cy,0,sy,  0,1,0,  -sy,0,cy );
    mat3 Rz = mat3( cz,-sz,0,  sz,cz,0,  0,0,1 );
    return Rz * Ry * Rx;
}

// finish: apply transforms, project, shadows, view-space N/T, UV
void end(inout Vertex v) {

    // Object-space transforms (hidden from effects)
    v.position.xyz = rot(v.position.xyz, v.angle);
    v.position.xyz += v.offset;

    // Projection to clip + view pos
    vertPosition = p3d_ModelViewMatrix * v.position;
    gl_Position  = p3d_ProjectionMatrix * vertPosition;

#ifdef OPOS
    opos = v.position.xyz;
#endif

    // Shadow spaces
#ifndef NO_SHADOWS
    for (int i = 0; i < u_NumberOfLights; ++i) {
        vertInShadowSpaces[i] = p3d_LightSource[i].shadowViewMatrix * vertPosition;
    }
#endif

    // View-space N/T with safe normalization and fallback
#ifndef NO_SHADOWS
    mat3  R  = rotm(v.angle);
    mat3  M  = mat3(p3d_NormalMatrix) * R;  // single matrix mul for both N & T
    vec3  Nv = M * p3d_Normal;
    vec3  Tv = M * p3d_Tangent.xyz;
    float invLenN = inversesqrt(max(dot(Nv, Nv), 1e-16));  // Normalize N (robust, branchless)
    Nv *= invLenN;
    vertNormal = Nv;  // Write varying (already normalized)
#endif

    // Normalize/repair T
#ifndef NO_TANGENT
    float lenT2 = dot(Tv, Tv);
    if (lenT2 < 1e-12) {
        // No tangent in mesh → build a temporary one orthogonal to N
        vec3 a  = (abs(Nv.z) < 0.999) ? vec3(0.0, 0.0, 1.0) : vec3(0.0, 1.0, 0.0);
        Tv = normalize(cross(a, Nv));
    } else {
        Tv *= inversesqrt(lenT2);
    }
    Tv -= Nv * dot(Tv, Nv);  // Gram–Schmidt: make T ⟂ N
    Tv *= inversesqrt(max(dot(Tv, Tv), 1e-16));  // Renormalize (cheap, stable)
    tangent = Tv;  // Write varying (already normalized)
#endif

    // Color
    vertColor = v.color;

    // UV
#ifndef NO_UV
    vertMultiTexCoord0 = p3d_MultiTexCoord0;
#endif

}

// ---- Hook (must be defined by effect file) ----
void effect(inout Vertex v);

// ---- Main ----
void main() {
    Vertex v; begin(v);
    effect(v);
    end(v);
}

#endif