
// epimgconv: Enterprise 128 image converter utility
// Copyright (C) 2008-2016 Istvan Varga <istvanv@users.sourceforge.net>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// The Enterprise 128 program files generated by this utility are not covered
// by the GNU General Public License, and can be used, modified, and
// distributed without any restrictions.

#include "epimgconv.hpp"
#include "imageconv.hpp"
#include "pixel4.hpp"

#include <vector>

namespace Ep128ImgConv {

  void ImageConv_Pixel4::initializePalettes()
  {
    for (int i = 0; i < 256; i++)
      convertEPColorToYUV(i, paletteY[i], paletteU[i], paletteV[i]);
  }

  void ImageConv_Pixel4::randomizePalette(int yc, int seedValue)
  {
    int     tmp = 0;
    setRandomSeed(tmp, uint32_t(seedValue));
    for (int i = 0; i < 4; i++) {
      unsigned char c = (unsigned char) (getRandomNumber(tmp) & 0xFF);
      if (!fixedColors[i])
        palette[yc][i] = c;
    }
  }

  double ImageConv_Pixel4::calculateLineError(int yc, double maxError)
  {
    float   tmpPaletteY[10];
    float   tmpPaletteU[10];
    float   tmpPaletteV[10];
    for (int i = 0; i < 4; i++) {
      tmpPaletteY[i] = paletteY[palette[yc][i]];
      tmpPaletteU[i] = paletteU[palette[yc][i]];
      tmpPaletteV[i] = paletteV[palette[yc][i]];
    }
    int     n = 4;
    for (int c0 = 0; c0 < 3; c0++) {
      for (int c1 = c0 + 1; c1 < 4; c1++) {
        tmpPaletteY[n] =
            0.5 * (paletteY[palette[yc][c0]] + paletteY[palette[yc][c1]]);
        tmpPaletteU[n] =
            0.5 * (paletteU[palette[yc][c0]] + paletteU[palette[yc][c1]]);
        tmpPaletteV[n] =
            0.5 * (paletteV[palette[yc][c0]] + paletteV[palette[yc][c1]]);
        n++;
      }
    }
    double  totalError = 0.0;
    float   tmpY = 0.0;
    float   tmpU = 0.0;
    float   tmpV = 0.0;
    for (int xc = 0; xc < width; xc++) {
      float   y = inputImage.y(xc, yc) + ditherErrorImage.y(xc, yc);
      float   u = inputImage.u(xc, yc) + ditherErrorImage.u(xc, yc);
      float   v = inputImage.v(xc, yc) + ditherErrorImage.v(xc, yc);
      limitYUVColor(y, u, v);
      tmpY += (y * 0.5f);
      tmpU += (u * 0.5f);
      tmpV += (v * 0.5f);
      double  minErr = 1000000000.0;
      for (int i = 0; i < 4; i++) {
        double  err = Ep128ImgConv::calculateYUVErrorSqr(tmpPaletteY[i],
                                                         tmpPaletteU[i],
                                                         tmpPaletteV[i],
                                                         y, u, v,
                                                         colorErrorScale);
        if (err < minErr)
          minErr = err;
      }
      totalError += minErr;
      if (xc & int(bool(ditherType))) {
        minErr = 1000000000.0;
        for (int i = 0; i < 10; i += 2) {
          double  err = Ep128ImgConv::calculateYUVErrorSqr(tmpPaletteY[i],
                                                           tmpPaletteU[i],
                                                           tmpPaletteV[i],
                                                           tmpY, tmpU, tmpV,
                                                           colorErrorScale);
          if (err < minErr)
            minErr = err;
          err = Ep128ImgConv::calculateYUVErrorSqr(tmpPaletteY[i + 1],
                                                   tmpPaletteU[i + 1],
                                                   tmpPaletteV[i + 1],
                                                   tmpY, tmpU, tmpV,
                                                   colorErrorScale);
          if (err < minErr)
            minErr = err;
        }
        totalError += (minErr * 4.0);
        tmpY = 0.0f;
        tmpU = 0.0f;
        tmpV = 0.0f;
      }
      if (totalError > (maxError * 1.00001))
        break;
    }
    return totalError;
  }

  double ImageConv_Pixel4::calculateTotalError(double maxError)
  {
    double  totalError = 0.0;
    for (int yc = 0; yc < height; yc++) {
      totalError += calculateLineError(yc);
      if (totalError > (maxError * 1.000001))
        break;
    }
    return totalError;
  }

  double ImageConv_Pixel4::optimizeLinePalette(int yc, int optimizeLevel)
  {
    double  bestError = 1000000000.0;
    int     bestPalette[4];
    for (int l = 0; l < optimizeLevel; l++) {
      randomizePalette(yc, l + 1);
      double  minErr = calculateLineError(yc);
      bool    doneFlag = true;
      do {
        doneFlag = true;
        for (int i = 0; i < 4; i++) {
          int     bestColor = palette[yc][i];
          if (!fixedColors[i]) {
            for (int c = 0; c < 256; c++) {
              palette[yc][i] = (unsigned char) c;
              double  err = calculateLineError(yc, minErr);
              if (err < (minErr * 0.999999)) {
                bestColor = c;
                doneFlag = false;
                minErr = err;
              }
            }
          }
          palette[yc][i] = (unsigned char) bestColor;
        }
      } while (!doneFlag);
      if (minErr < bestError) {
        for (int i = 0; i < 4; i++)
          bestPalette[i] = palette[yc][i];
        bestError = minErr;
      }
    }
    for (int i = 0; i < 4; i++)
      palette[yc][i] = (unsigned char) bestPalette[i];
    sortLinePalette(yc);
    return bestError;
  }

  double ImageConv_Pixel4::optimizeImagePalette(int optimizeLevel)
  {
    double  bestError = 1000000000.0;
    int     bestPalette[4];
    int     progressCnt = 0;
    int     progressMax = optimizeLevel * 3 * 4 * 256;
    for (int l = 0; l < optimizeLevel; l++) {
      randomizePalette(0, l + 1);
      setFixedPalette();
      double  minErr = calculateTotalError();
      bool    doneFlag = true;
      int     loopCnt = 0;
      do {
        doneFlag = true;
        if (++loopCnt > 3)
          progressCnt -= (4 * 256);
        for (int i = 0; i < 4; i++) {
          int     bestColor = palette[0][i];
          if (!fixedColors[i]) {
            for (int c = 0; c < 256; c++) {
              if (!setProgressPercentage(progressCnt * 100 / progressMax))
                return -1.0;
              progressCnt++;
              palette[0][i] = (unsigned char) c;
              setFixedPalette();
              double  err = calculateTotalError(minErr);
              if (err < (minErr * 0.999999)) {
                bestColor = c;
                doneFlag = false;
                minErr = err;
              }
            }
          }
          else {
            if (!setProgressPercentage(progressCnt * 100 / progressMax))
              return -1.0;
            progressCnt += 256;
          }
          palette[0][i] = (unsigned char) bestColor;
        }
      } while (!doneFlag);
      if (loopCnt < 3)
        progressCnt += ((3 - loopCnt) * (4 * 256));
      if (minErr < bestError) {
        for (int i = 0; i < 4; i++)
          bestPalette[i] = palette[0][i];
        bestError = minErr;
      }
    }
    for (int i = 0; i < 4; i++)
      palette[0][i] = (unsigned char) bestPalette[i];
    sortLinePalette(0);
    setFixedPalette();
    return bestError;
  }

  void ImageConv_Pixel4::sortLinePalette(int yc)
  {
    // sort palette colors by bit-reversed color value
    for (int i = 0; i < 4; i++) {
      unsigned char tmp = palette[yc][i];
      tmp = ((tmp & 0x55) << 1) | ((tmp & 0xAA) >> 1);
      tmp = ((tmp & 0x33) << 2) | ((tmp & 0xCC) >> 2);
      tmp = ((tmp & 0x0F) << 4) | ((tmp & 0xF0) >> 4);
      palette[yc][i] = tmp;
    }
    for (int i = 0; i < 3; i++) {
      for (int j = i + 1; j < 4; j++) {
        if (fixedColors[i] || fixedColors[j] ||
            palette[yc][i] <= palette[yc][j]) {
          continue;
        }
        unsigned char tmp = palette[yc][i];
        palette[yc][i] = palette[yc][j];
        palette[yc][j] = tmp;
      }
    }
    for (int i = 0; i < 4; i++) {
      unsigned char tmp = palette[yc][i];
      tmp = ((tmp & 0x55) << 1) | ((tmp & 0xAA) >> 1);
      tmp = ((tmp & 0x33) << 2) | ((tmp & 0xCC) >> 2);
      tmp = ((tmp & 0x0F) << 4) | ((tmp & 0xF0) >> 4);
      palette[yc][i] = tmp;
    }
  }

  void ImageConv_Pixel4::setFixedPalette()
  {
    for (int yc = 1; yc < height; yc++) {
      for (int i = 0; i < 4; i++)
        palette[yc][i] = palette[0][i];
    }
  }

  void ImageConv_Pixel4::pixelStoreCallback(void *userData, int xc, int yc,
                                            float y, float u, float v)
  {
    ImageConv_Pixel4&  this_ =
        *(reinterpret_cast<ImageConv_Pixel4 *>(userData));
    xc = xc >> 1;
    yc = yc >> 1;
    if (xc < 0 || xc >= this_.width || yc < 0 || yc >= this_.height)
      return;
    limitYUVColor(y, u, v);
    this_.inputImage.y(xc, yc) += (y * 0.25f);
    this_.inputImage.u(xc, yc) += (u * 0.25f);
    this_.inputImage.v(xc, yc) += (v * 0.25f);
  }

  void ImageConv_Pixel4::pixelStoreCallbackI(void *userData, int xc, int yc,
                                             float y, float u, float v)
  {
    ImageConv_Pixel4&  this_ =
        *(reinterpret_cast<ImageConv_Pixel4 *>(userData));
    xc = xc >> 1;
    if (xc < 0 || xc >= this_.width || yc < 0 || yc >= this_.height)
      return;
    limitYUVColor(y, u, v);
    this_.inputImage.y(xc, yc) += (y * 0.5f);
    this_.inputImage.u(xc, yc) += (u * 0.5f);
    this_.inputImage.v(xc, yc) += (v * 0.5f);
  }

  ImageConv_Pixel4::ImageConv_Pixel4()
    : ImageConverter(),
      width(1),
      height(1),
      colorErrorScale(0.5f),
      inputImage(1, 1),
      ditherErrorImage(1, 1),
      convertedImage(1, 1),
      palette(4, 1),
      conversionQuality(3),
      borderColor(0x00),
      ditherType(1),
      ditherDiffusion(0.95f)
  {
    for (int i = 0; i < 4; i++)
      fixedColors[i] = false;
    initializePalettes();
  }

  ImageConv_Pixel4::~ImageConv_Pixel4()
  {
  }

  bool ImageConv_Pixel4::processImage(
      ImageData& imgData, const char *infileName,
      YUVImageConverter& imgConv, const ImageConvConfig& config)
  {
    width = config.width << 3;
    height = ((imgData[5] & 0x80) == 0 ? config.height : (config.height << 1));
    colorErrorScale = float(config.colorErrorScale);
    conversionQuality = config.conversionQuality;
    borderColor = config.borderColor & 0xFF;
    float   borderY = 0.0f;
    float   borderU = 0.0f;
    float   borderV = 0.0f;
    convertEPColorToYUV(borderColor, borderY, borderU, borderV);
    inputImage.setBorderColor(borderY, borderU, borderV);
    ditherType = config.ditherType;
    limitValue(ditherType, 0, 5);
    ditherDiffusion = float(config.ditherDiffusion);

    inputImage.resize(width, height);
    ditherErrorImage.resize(width, height);
    convertedImage.resize(width, height);
    palette.resize(4, height);
    inputImage.clear();
    ditherErrorImage.clear();
    convertedImage.clear();
    palette.clear();

    initializePalettes();
    for (int i = 0; i < 4; i++)
      fixedColors[i] = (config.paletteColors[i] >= 0);
    for (int yc = 0; yc < height; yc++) {
      randomizePalette(yc, yc + 1000);
      for (int i = 0; i < 4; i++) {
        if (fixedColors[i])
          palette[yc][i] = (unsigned char) (config.paletteColors[i] & 0xFF);
      }
    }

    if (!(imgData[5] & 0x80))
      imgConv.setPixelStoreCallback(&pixelStoreCallback, (void *) this);
    else
      imgConv.setPixelStoreCallback(&pixelStoreCallbackI, (void *) this);
    if (!imgConv.convertImageFile(infileName))
      return false;

    progressMessage("Converting image");
    setProgressPercentage(0);
    int     optimizeLevel = 1 + ((conversionQuality - 1) >> 1);
    ditherErrorImage.clear();
    if (config.paletteResolution != 0) {
      // generate optimal palette independently for each line
      int     progressCnt = 0;
      int     progressMax = height;
      for (int yc = 0; yc < height; yc++) {
        if (!setProgressPercentage((progressCnt * 100) / progressMax))
          return false;
        optimizeLinePalette(yc, optimizeLevel);
        ditherLine(convertedImage, inputImage, ditherErrorImage, yc,
                   ditherType, ditherDiffusion,
                   colorErrorScale, &(palette[yc][0]), 4,
                   paletteY, paletteU, paletteV);
        progressCnt++;
      }
    }
    else {
      // generate optimal palette for the whole image
      if (optimizeImagePalette(optimizeLevel) < -0.5)
        return false;
      for (int yc = 0; yc < height; yc++) {
        ditherLine(convertedImage, inputImage, ditherErrorImage, yc,
                   ditherType, ditherDiffusion,
                   colorErrorScale, &(palette[0][0]), 4,
                   paletteY, paletteU, paletteV);
      }
    }
    imgData.setBorderColor(borderColor);
    for (int yc = 0; yc < height; yc++) {
      for (int i = 0; i < 4; i++)
        imgData.setPaletteColor(yc, i, palette[yc][i]);
      for (int xc = 0; xc < width; xc++)
        imgData.setPixel(xc, yc, convertedImage[yc][xc]);
    }
    setProgressPercentage(100);
    if (config.paletteResolution != 0) {
      progressMessage("");
    }
    else {
      char    tmpBuf[64];
      std::sprintf(&(tmpBuf[0]), "Done; palette = %d %d %d %d",
                   int(palette[0][0]), int(palette[0][1]),
                   int(palette[0][2]), int(palette[0][3]));
      progressMessage(&(tmpBuf[0]));
    }
    return true;
  }

}       // namespace Ep128ImgConv

