// vgm2nes
// Mic, 2011

#include <cstdlib>
#include <iostream>
#include <vector>
#include <string>
#include <math.h>
#include <time.h>
#include <zlib.h>
#include "nullcodec.h"


#define DEFAULT_PSG_CLOCK 3579545
#define NES_PAL_CLOCK 1662607.0f
#define NES_NTSC_CLOCK 1789772.0f

#define PLAYER_NTSC_TABLES 0x5010
#define PLAYER_PAL_TABLES 0x7010
#define PLAYER_COMMON_TABLES 0x3FD10

using namespace std;

const float NES_NOISE_FREQ[16] = 
{
	447443, 223721, 118860, 55930,
	27965, 18643, 13982, 11186,
	8860, 7046, 4709, 3523,
	2348, 1761, 880, 440
};


int *destLen_;
unsigned char *outBuffer_;
FILE *outFile_;

void set_output(FILE *fp, unsigned char *ob, int *dl)
{
	outFile_ = fp;
	outBuffer_ = ob;
	destLen_ = dl;
}

void wrap_fputc(int ch, FILE *fp)
{
	if (destLen_ == NULL)
	{
		fputc(ch, fp);
	}
	else
	{
		outBuffer_[(*destLen_)++] = (unsigned char)ch;
	}
}

void wrap_fprint(FILE *fp, char *str)
{
	if (destLen_ == NULL)
	{
		fprintf(fp, str);
	}
	else
	{
		for (int i = 0; i < strlen(str); i++)
		{
			outBuffer_[(*destLen_)++] = str[i];
		}
	}
}

void wrap_fwrite(char *str, int s, int len, FILE *fp)
{
	if (destLen_ == NULL)
	{
		fwrite(str, s, len, fp);
	}
	else
	{
		for (int i = 0; i < len; i++)
		{
			outBuffer_[(*destLen_)++] = str[i];
		}
	}
}



void write_and_pad(FILE *fp, char *str, int len)
{
	int sl = strlen(str);
	int wl = (sl <= len) ? sl : len;

	wrap_fwrite(str, 1, wl, fp);
	for (; wl < len; wl++) wrap_fputc(0, fp);
}


void pack(char *inFileName, char *outFileName, int *destLen, int mode)
{
    unsigned int i, inSize, origInSize, vgmOffs, preOffs, loopOffs, newLoopOffs, progress,
		         extraDataOffs, lastPcmOffs, unzippedBytes, playerSize, rawOutput;
	FILE *inFile, *outFile, *playerFile;
	gzFile inFileZ;
	unsigned char c, ch3Mode = 0;
	unsigned char *vgmData, *preProcessedData, *outBuffer;
	vector<unsigned char> outData;
	vector<unsigned char> extraDataBlock;
	vector<unsigned char> unzippedData;
	unsigned short *longWaitLut;
	Codec *codec;
	bool endOfVgmData = false;
	bool vgz = false;
	static char game[32] = "Unknown", song[32] = "Unknown", artist[32] = "Unknown";
	static char dateDumped[11] = "01/01/1990";
	static const char vgmSignature[] = "Vgm ";
	static unsigned char unzipBuffer[16384];
	int toneFreq[2][1024], perNoiseFreq[2][1024], whiteNoiseFreq[256];
	int psgClock;

	rawOutput = mode & 8;
	mode &= 7;

	if (mode & 4)
	{
		vgz = true;
		mode &= 3;
	}
	else
	{
		if ((inFileName[strlen(inFileName) - 1] == 'z') || (inFileName[strlen(inFileName) - 1] == 'Z'))
		{
			vgz = true;
		}
	}

	// Try to auto-detect .vgz files that have been named .vgm based on the first four bytes
	if (!vgz)
	{
		fopen_s(&inFile, inFileName, "rb");
		if (inFile == NULL)
		{
			printf("Error: Failed to open %s\n", inFileName);
			exit(0);
		}
		for (i = 0; i < 4; i++)
		{
			if (getc(inFile) != vgmSignature[i])
			{
				vgz = true;
				break;
			}
		}
		fclose(inFile);
	}

	if (!vgz)
	{
		fopen_s(&inFile, inFileName, "rb");
		if (inFile == NULL)
		{
			printf("Error: Failed to open %s\n", inFileName);
			exit(0);
		}
		fseek(inFile, 0, SEEK_END);
		inSize = origInSize = ftell(inFile);
		fseek(inFile, 0, SEEK_SET);

		if ((vgmData = new unsigned char[inSize]) == NULL)
		{
			puts("Error: Failed to allocate memory");
			return;
		}
		if ((preProcessedData = new unsigned char[inSize]) == NULL)
		{
			puts("Error: Failed to allocate memory");
			return;
		}
		if (fread(vgmData, 1, inSize, inFile) != inSize)
		{
			puts("Error: Failed to read VGM data");
			exit(0);
		}
		fclose(inFile);
	}
	else
	{
		puts("Unzipping file");
		inFileZ = gzopen(inFileName, "rb");
		if (inFileZ == NULL)
		{
			printf("Error: Failed to gzopen %s\n", inFileName);
			exit(0);
		}
		while (true)
		{
			unzippedBytes = gzread(inFileZ, unzipBuffer, 16384);
			if (unzippedBytes > 0)
			{
				for (i = 0; i < unzippedBytes; i++)
				{
					unzippedData.push_back(unzipBuffer[i]);
				}
			}
			else
			{
				break;
			}
		}
		gzclose(inFileZ);
		inSize = origInSize = (unsigned int)unzippedData.size();
		if ((vgmData = new unsigned char[inSize]) == NULL)
		{
			printf("Error: Failed to allocate memory (wanted %d bytes)\n", inSize);
			return;
		}
		for (i = 0; i < inSize; i++)
		{
			vgmData[i] = unzippedData[i];
		}
		unzippedData.clear();	
		preProcessedData = new unsigned char[inSize];
	}

	if (vgmData[8] < 0x10)
	{
		vgmOffs = 0x40;
	}
	else
	{
		vgmOffs = vgmData[0x34] +
				  (vgmData[0x35] << 8) +
				  (vgmData[0x36] << 16) +
				  (vgmData[0x37] << 24) + 0x34;
		if (vgmOffs < 0x40) vgmOffs = 0x40;
	}
	extraDataOffs = vgmOffs;

	psgClock = vgmData[0x0C] +
			   (vgmData[0x0D] << 8) +
			   (vgmData[0x0E] << 16) +
			   (vgmData[0x0F] << 24);

	if (psgClock < 1000000 || psgClock > 7000000)
		psgClock = DEFAULT_PSG_CLOCK;

	if (psgClock != DEFAULT_PSG_CLOCK)
	{
		printf("Generating tables based on %d Hz clock\n", psgClock);

		for (i = 0; i < 1024; i++)
		{
			float divider = i;
			if (divider == 0) divider = 1;
			float freq = (float)psgClock / (2.0f * divider * 16.0f);
			float ntscP = NES_NTSC_CLOCK / (16.0f * freq) - 1;
			float palP = NES_PAL_CLOCK / (16.0f * freq) - 1;
			if (ntscP > 4095) ntscP = 4095;
			if (palP > 4095) palP = 4095;
			toneFreq[0][i] = (int)ntscP;
			toneFreq[1][i] = (int)palP;
		}

		for (i = 0; i < 1024; i++)
		{
			float divider = i;
			if (divider == 0) divider = 1;
			float freq = (float)psgClock / (16.0f * divider * 32.0f);
			float ntscP = NES_NTSC_CLOCK / (16.0f * freq) - 1;
			float palP = NES_PAL_CLOCK / (16.0f * freq) - 1;
			if (ntscP > 4095) ntscP = 4095;
			if (palP > 4095) palP = 4095;
			perNoiseFreq[0][i] = (int)ntscP;
			perNoiseFreq[1][i] = (int)palP;
		}

		float diff = 1000000;
		int bestMatch = 0;
		for (i = 0; i < 256; i++)
		{
			float divider = i;
			if (divider == 0) divider = 1;
			float freq = (float)psgClock / (16.0f * divider * 32.0f);
			diff = 1000000;
			for (int j = 0; j < 16; j++)
			{
				if (fabs(freq - NES_NOISE_FREQ[j]/16.0f) < diff) 
				{
					bestMatch = j;
					diff = fabs(freq - NES_NOISE_FREQ[j]/16.0f);
				}
			}
			whiteNoiseFreq[i] = bestMatch;
		}
	}


	loopOffs = vgmData[0x1C] +
		       (vgmData[0x1D] << 8) +
			   (vgmData[0x1E] << 16) +
			   (vgmData[0x1F] << 24) + 0x1C;
	newLoopOffs = loopOffs;

	vgmData[8] = 0x52;	// To identify the VGM as compressed

	printf("Converting %s:", inFileName);

	for (i = 0; i < vgmOffs; i++)
	{
		outData.push_back(vgmData[i]);
	}

	codec = new NullCodec(&outData);

	for (i = 0; i < vgmOffs; i++)
	{
		preProcessedData[i] = vgmData[i];
	}
	preOffs = vgmOffs;
	lastPcmOffs = 0xFFFFFFFF;

	// Run a pre-processing stage to remove redundant commands
	while (!endOfVgmData)
	{
		if (vgmOffs == loopOffs)
		{
			newLoopOffs = preOffs;
			//printf("[1] loop %x -> %x\n", loopOffs, newLoopOffs);
		}

		c = vgmData[vgmOffs++];

		switch (c)
		{
			case 0x52:
				c = vgmData[vgmOffs++];
				if (c == 0x27)
				{
					c = vgmData[vgmOffs++];
					if ((c >> 6) != ch3Mode)
					{
						ch3Mode = c >> 6;
						preProcessedData[preOffs++] = 0x52;
						preProcessedData[preOffs++] = 0x27;
						preProcessedData[preOffs++] = c;
					}
				}
				else if ((c == 0x25) || (c == 0x26))
				{
					vgmOffs++;
				}
				else
				{
					preProcessedData[preOffs++] = 0x52;
					preProcessedData[preOffs++] = c;
					preProcessedData[preOffs++] = vgmData[vgmOffs++];
				}
				break;

			case 0x51:
			case 0x53:
			case 0x54:
			case 0x61:
			case 0xB3:
			case 0xB4:
				preProcessedData[preOffs++] = c;
				preProcessedData[preOffs++] = vgmData[vgmOffs++];
				preProcessedData[preOffs++] = vgmData[vgmOffs++];
				break;

			case 0x66:
				preProcessedData[preOffs++] = c;
				endOfVgmData = true;
				break;

			case 0x67:
				if (vgmData[vgmOffs] == 0x66)
				{
					unsigned int dataSize = vgmData[vgmOffs+2];
					dataSize += vgmData[vgmOffs+3] << 8;
					dataSize += vgmData[vgmOffs+4] << 16;
					dataSize += vgmData[vgmOffs+5] << 24;
					preProcessedData[preOffs++] = c;
					for (i = 0; i < dataSize + 6; i++)
					{
						preProcessedData[preOffs++] = vgmData[vgmOffs++];
					}
				}
				else
				{
					printf("Illegal command: 0x67 0x%02x\n", vgmData[vgmOffs]);
					delete [] vgmData;
					delete [] preProcessedData;
					delete codec;
					exit(0);
				}
				break;

			case 0xE0:
				i = vgmData[vgmOffs++];
				i += vgmData[vgmOffs++] << 8;
				i += vgmData[vgmOffs++] << 16;
				i += vgmData[vgmOffs++] << 24;
				if ((mode == 0) || (i != 0))
				{
					preProcessedData[preOffs++] = c;
					preProcessedData[preOffs++] = i & 0xFF;
					preProcessedData[preOffs++] = (i >> 8) & 0xFF;
					preProcessedData[preOffs++] = (i >> 16) & 0xFF;
					preProcessedData[preOffs++] = (i >> 24) & 0xFF;
					lastPcmOffs = i;
				}
				else if (mode != 0)
				{
					preProcessedData[preOffs++] = 0x4C;
				}
				break;

			case 0x80: case 0x81: case 0x82: case 0x83: case 0x84: case 0x85: case 0x86: case 0x87:
			case 0x88: case 0x89: case 0x8A: case 0x8B: case 0x8C: case 0x8D: case 0x8E: case 0x8F:
				c &= 0x0F;
				if ((vgmData[vgmOffs] & 0xF0) == 0x70)
				{
					while ((vgmData[vgmOffs] & 0xF0) == 0x70)
					{
						if ((c + (vgmData[vgmOffs] & 0x0F)) < 0x10)
						{
							c += vgmData[vgmOffs] & 0x0F;
							vgmOffs++;
						}
						else
						{
							break;
						}
					}
					preProcessedData[preOffs++] = 0x80 | c;
				}
				else if (preOffs > 0)
				{
					if ((preProcessedData[preOffs - 1] & 0xF0) == 0x70)
					{
						if ((c + (preProcessedData[preOffs - 1] & 0x0F)) < 0x10)
						{
							c += preProcessedData[preOffs - 1] & 0x0F;
						}
						preProcessedData[preOffs - 1] = 0x80 | c;
					}
					else
					{
						preProcessedData[preOffs++] = 0x80 | c;
					}
				}
				else
				{
					preProcessedData[preOffs++] = 0x80 | c;
				}
				break;

			default:
				preProcessedData[preOffs++] = c;
				break;

		}
	}
	while (vgmOffs < inSize)
	{
		preProcessedData[preOffs++] = vgmData[vgmOffs++];
	}

	loopOffs = newLoopOffs;
	vgmOffs = extraDataOffs;
	endOfVgmData = false;
	delete [] vgmData;
	vgmData = preProcessedData;
	inSize = preOffs;

	progress = 0;

	// Now do the encoding stage
	while (!endOfVgmData)
	{
		if (((vgmOffs * 10) / inSize) > progress)
		{
			progress++;
			printf("%c", 177);
		}

		if (vgmOffs == loopOffs)
		{
			newLoopOffs = (unsigned int)outData.size();
		}

		c = vgmData[vgmOffs++];

		if (c != 0x67)
			codec->write(c);

		switch (c)
		{
			case 0x4F: case 0x30:
				codec->arg(vgmData[vgmOffs++]);
				break;
			case 0x50:
				codec->arg(vgmData[vgmOffs++]);
				break;

			case 0x51:
			case 0x52:
			case 0x53:
			case 0x54:
			case 0x61:
			case 0xB3:
			case 0xB4:
				codec->arg(vgmData[vgmOffs++]);
				codec->arg(vgmData[vgmOffs++]);
				break;

			case 0x66:
				codec->flush();
				endOfVgmData = true;
				break;

			case 0x67:
				if (vgmData[vgmOffs] == 0x66)
				{
					unsigned int dataSize = vgmData[vgmOffs+2];
					dataSize += vgmData[vgmOffs+3] << 8;
					dataSize += vgmData[vgmOffs+4] << 16;
					dataSize += vgmData[vgmOffs+5] << 24;
					// Skip data blocks
					for (i = 0; i < dataSize + 6; i++)
					{
						vgmOffs++;
						//codec->arg(vgmData[vgmOffs++]);
					}
				}
				else
				{
					printf("Illegal command: 0x67 0x%02x\n", vgmData[vgmOffs]);
					delete [] vgmData;
					delete codec;
					exit(0);
				}
				break;

			case 0xE0:
				codec->arg(vgmData[vgmOffs++]);
				codec->arg(vgmData[vgmOffs++]);
				codec->arg(vgmData[vgmOffs++]);
				codec->arg(vgmData[vgmOffs++]);
				break;
		}
		if ((outData.size() & 0x3FFF) > 0x3FFC)
		{
			codec->write(0x4B);		// bankswitch command
			while (outData.size() & 0x3FFF) codec->write(0);
		}
	}

	delete codec;

	// EOF offset
	i = (unsigned int)(outData.size() + extraDataBlock.size() - 4);
	outData[4] = i & 0xFF;
	outData[5] = (i >> 8) & 0xFF;
	outData[6] = (i >> 16) & 0xFF;
	outData[7] = (i >> 24) & 0xFF;

	// Read rest of data, if any (GD3)
	while (vgmOffs < inSize)
	{
		outData.push_back(vgmData[vgmOffs++]);
	}

	// GD3 offset
	i = outData[0x14] +
		(outData[0x15] << 8) +
		(outData[0x16] << 16) +
		(outData[0x17] << 24);
	if (i)
	{
		i -= (origInSize - (outData.size() + extraDataBlock.size()));
		outData[0x14] = i & 0xFF;
		outData[0x15] = (i >> 8) & 0xFF;
		outData[0x16] = (i >> 16) & 0xFF;
		outData[0x17] = (i >> 24) & 0xFF;

		// Read song title, game title and author from the GD3 header and convert from
		// wide chars to char.
		int j,k;
		j = i + 0x14 + 0x0C - extraDataBlock.size();
		for (k = 0; k < 32; k++)
		{
			if (outData[j] == 0) break;
			song[k] = outData[j];
			j += 2;
		}
		if (k > 0 && k < 32) song[k] = 0;
		while (outData[j] != 0) j += 2;		// Skip the rest of the title
		j += 2;

		while (outData[j] != 0) j += 2;		// Skip japanese title
		j += 2;

		for (k = 0; k < 32; k++)
		{
			if (outData[j] == 0) break;
			game[k] = outData[j];
			j += 2;
		}
		if (k > 0 && k < 32) game[k] = 0;
		while (outData[j] != 0) j += 2;		// Skip the rest of the game title
		j += 2;
		while (outData[j] != 0) j += 2;		// Skip japanese game title
		j += 2;

		while (outData[j] != 0) j += 2;		// Skip system name
		j += 2;	
		while (outData[j] != 0) j += 2;		// Skip japanese system name
		j += 2;

		for (k = 0; k < 32; k++)
		{
			if (outData[j] == 0) break;
			artist[k] = outData[j];
			j += 2;
		}
		if (k > 0 && k < 32) artist[k] = 0;
	}

	// Loop offset
	if (loopOffs > 0x1C)
	{
		newLoopOffs += (unsigned int)extraDataBlock.size();
		newLoopOffs -= 0x1C;
		outData[0x1C] = newLoopOffs & 0xFF;
		outData[0x1D] = (newLoopOffs >> 8) & 0xFF;
		outData[0x1E] = (newLoopOffs >> 16) & 0xFF;
		outData[0x1F] = (newLoopOffs >> 24) & 0xFF;
	}

	outData[0x34] = (0x8000 + extraDataOffs) & 0xFF;
	outData[0x35] = ((0x8000 + extraDataOffs)>>8) & 0xFF;
	outData[0x36] = ((0x8000 + extraDataOffs)>>16) & 0xFF;
	outData[0x37] = ((0x8000 + extraDataOffs)>>24) & 0xFF;

	outData[0x38] = (0x8000 + newLoopOffs + 0x1C) & 0xFF;
	outData[0x39] = ((0x8000 + newLoopOffs + 0x1C)>>8) & 0xFF;
	outData[0x3A] = ((0x8000 + newLoopOffs + 0x1C)>>16) & 0xFF;
	outData[0x3B] = ((0x8000 + newLoopOffs + 0x1C)>>24) & 0xFF;

	while (outData.size() & 0x7FFF) outData.push_back(0);

	while (progress < 10)
	{
		printf("%c\n", 177);
		progress++;
	}

	// Open the player binary
	fopen_s(&playerFile, "player.nes", "rb");

	if (playerFile == NULL)
	{
		delete [] vgmData;
		outData.clear();
		printf("Error: unable to open %s\n", "player.nes");
		return;
	}
	fseek(playerFile, 0, SEEK_END);
	playerSize = ftell(playerFile);
	fseek(playerFile, 0, SEEK_SET);

	if (destLen == NULL)
	{
		fopen_s(&outFile, outFileName, "wb");
	}
	else
	{
		outBuffer = (unsigned char*)outFileName;
		outFile = (FILE*)1;
	}

	set_output(outFile, outBuffer, destLen);

	if (outFile == NULL)
	{
		delete [] vgmData;
		printf("Error: unable to open %s\n", outFileName);
	}
	else
	{

		{
			for (i = 0x0; i < 0x8000+16; i++)
			{
				int ch = fgetc(playerFile);
				if (ch == EOF) break;

				if (psgClock != DEFAULT_PSG_CLOCK)
				{
					if (i >= PLAYER_NTSC_TABLES-0x800 && i < PLAYER_NTSC_TABLES+6-0x800)
					{
						if (i & 1)
							ch = perNoiseFreq[0][8 << ((i - (PLAYER_NTSC_TABLES-0x800))>>1)] >> 8;
						else
							ch = perNoiseFreq[0][8 << ((i - (PLAYER_NTSC_TABLES-0x800))>>1)] & 0xFF;
					}
					else if (i >= PLAYER_PAL_TABLES-0x800 && i < PLAYER_PAL_TABLES+6-0x800)
					{
						if (i & 1)
							ch = perNoiseFreq[1][8 << ((i - (PLAYER_PAL_TABLES-0x800))>>1)] >> 8;
						else
							ch = perNoiseFreq[1][8 << ((i - (PLAYER_PAL_TABLES-0x800))>>1)] & 0xFF;
					}
					else if (i >= PLAYER_NTSC_TABLES && i < PLAYER_NTSC_TABLES+0x400)
					{
						ch = toneFreq[0][i - PLAYER_NTSC_TABLES] & 0xFF;
					}
					else if (i >= PLAYER_NTSC_TABLES+0x400 && i < PLAYER_NTSC_TABLES+0x800)
					{
						ch = toneFreq[0][i - (PLAYER_NTSC_TABLES+0x400)] >> 8;
					}
					else if (i >= PLAYER_NTSC_TABLES+0x800 && i < PLAYER_NTSC_TABLES+0xC00)
					{
						ch = perNoiseFreq[0][i - (PLAYER_NTSC_TABLES+0x800)] & 0xFF;
					}
					else if (i >= PLAYER_NTSC_TABLES+0xC00 && i < PLAYER_NTSC_TABLES+0x1000)
					{
						ch = perNoiseFreq[0][i - (PLAYER_NTSC_TABLES+0xC00)] >> 8;
					}
					else if (i >= PLAYER_PAL_TABLES && i < PLAYER_PAL_TABLES+0x400)
					{
						ch = toneFreq[1][i - PLAYER_PAL_TABLES] & 0xFF;
					}
					else if (i >= PLAYER_PAL_TABLES+0x400 && i < PLAYER_PAL_TABLES+0x800)
					{
						ch = toneFreq[1][i - (PLAYER_PAL_TABLES+0x400)] >> 8;
					}
					else if (i >= PLAYER_PAL_TABLES+0x800 && i < PLAYER_PAL_TABLES+0xC00)
					{
						ch = perNoiseFreq[1][i - (PLAYER_PAL_TABLES+0x800)] & 0xFF;
					}
					else if (i >= PLAYER_PAL_TABLES+0xC00 && i < PLAYER_PAL_TABLES+0x1000)
					{
						ch = perNoiseFreq[1][i - (PLAYER_PAL_TABLES+0xC00)] >> 8;
					}
				}
				wrap_fputc(ch, outFile);
			}
			for (; i < 0x8000+16; i++) wrap_fputc(0, outFile);
		}
		//fclose(playerFile);

		for (i = 0; i < extraDataOffs; i++)
		{
			wrap_fputc(outData[i], outFile);
		}
		for (unsigned int j = 0; j < extraDataBlock.size(); j++)
		{
			wrap_fputc(extraDataBlock[j], outFile);
		}
		for (; i < outData.size(); i++)
		{
			wrap_fputc(outData[i], outFile);
		}
		
		i = 0x8000+16 + outData.size();
		for (; i < 248*1024+16; i++)
		{
			wrap_fputc(0, outFile);
		}

		// Last PRG bank + CHR bank
		fseek(playerFile, 248*1024+16, SEEK_SET);
		for (i = 0; i < 2*8*1024; i++)
		{
			int ch = fgetc(playerFile);
			if (i >= (PLAYER_COMMON_TABLES-0x3E010) && i < (PLAYER_COMMON_TABLES+256-0x3E010))
			{
				ch = whiteNoiseFreq[i - (PLAYER_COMMON_TABLES-0x3E010)];
			}
			wrap_fputc(ch, outFile);
		}
		fclose(playerFile);

		if (destLen == NULL)
		{
			fclose(outFile);
		}
	}

	delete [] vgmData;
	outData.clear();

	puts("Done!");
}





int main(int argc, char *argv[])
{
	int i, mode = 0;
	char *inFn = NULL, *outFn = NULL;

	puts("VGM to NES Converter by Mic, 2011");

	for (i = 1; i < argc; i++)
	{
		if (argv[i][0] == '-')
		{
			if ((strcmp(argv[i], "-h")==0) ||
				(strcmp(argv[i], "-help")==0) ||
				(strcmp(argv[i], "-?")==0))
			{
				printf("Usage: vgm2nes <input> <output>\n");
				return 0;
			}
		}
		else if (inFn == NULL)
		{
			inFn = argv[i];
			if (inFn[0] == '\"')
			{
				inFn[strlen(inFn) - 1] = '\0';
				inFn++;
			}
		}
		else if (outFn == NULL)
		{
			outFn = argv[i];
			if (outFn[0] == '\"')
			{
				outFn[strlen(outFn) - 1] = '\0';
				outFn++;
			}
		}
	}

	if ((inFn == NULL) || (outFn == NULL))
	{
		printf("Usage: vgm2nes <input> <output>\n");
        return 0;
    }

	pack(inFn, outFn, NULL, mode);

	return 0;
}


